
/*
   dbgdev.c

   character device for debuging. 
   our board don't have a console, 
   so we have to redirrect all kinds of output data to somewhere
   meaningful. with this device, warning and error reporting can 
   be put into it and later retrieved by some application programs.

   all data should be write into this file with a prefix indicating
   its warning level. if not present, we will think it's the lower
   level, that is, Debug info.


modification history   
--
   01a, 08Mar2005, lie created.
*/

#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/fs.h>
#include <linux/sched.h>
#include <linux/wait.h>
#include <linux/init.h>
#include <asm/uaccess.h>


MODULE_LICENSE("GPL");

#define DEVICE_NAME "dbgdev"
#define DBGDEV_MAJOR  254

#define DEBUG 0

#if DEBUG
#define TRACE printk
#else
#define TRACE
#endif

static ssize_t device_read(struct file *f, char *buf, size_t buf_size, loff_t *off);
static ssize_t device_write(struct file *f, const char* buf, size_t buf_size, loff_t *off);
static int device_open(struct inode *node, struct file *f);
static int device_release(struct inode *node, struct file *f);
static int device_ioctl(struct inode *node, struct file *f, unsigned int command, unsigned long value);

// global variables
static struct file_operations fops={
  read: device_read,
  write: device_write,
  open : device_open,
  release:  device_release,
  ioctl: device_ioctl
};

#define MSG_BUF_SIZE 100
#define MSG_NUM      128 //should be power of 2.
#define MSG_NUM_MASK (MSG_NUM - 1)

typedef struct msg_node_t {
	int type;
	size_t size;
	char buf[MSG_BUF_SIZE + 4];
}msg_node;

typedef struct dbg_data {
	msg_node msg[MSG_NUM];
	int wr;
	int rd;
	int warn_level; //only higher level messages will be stored!
	int output_to_console;
	wait_queue_head_t dbgdev_wq;
}dbg_data;

static dbg_data all_dbg_data;

static int ref_cnt;

static int __init dbgdev_init(void)
{
	int ret;
	ret = register_chrdev(DBGDEV_MAJOR, DEVICE_NAME, &fops);

	if (ret >= 0) {
		printk("debug device (%s) registered, ret = %d\r\n", DEVICE_NAME, ret);
	} else {
		printk("debug device (%s) failed to register(%d).\r\n", DEVICE_NAME, ret);
	}

	ref_cnt = 0;
	init_waitqueue_head(&all_dbg_data.dbgdev_wq);
	all_dbg_data.warn_level = 2;
	all_dbg_data.output_to_console = 0;
	return 0;
}

static void __exit dbgdev_cleanup(void)
{
	int ret = unregister_chrdev(DBGDEV_MAJOR, DEVICE_NAME);
	printk("<1>%s relased: %d\r\n", DEVICE_NAME, ret);
}


static ssize_t device_read(struct file *f, char *buf, size_t buf_size, loff_t* off)
{
	int count;
	dbg_data *data  = f->private_data;
	
	TRACE("%s: read %d, write %d, buf_size %d\r\n",
		DEVICE_NAME, data->rd, data->wr, buf_size);
	//this while checking is not necessary since the wait-event-interruptable can do this check.
	while (data->rd == data->wr) {
		TRACE("dbgdev: no data available, sleep. \n");
		if(wait_event_interruptible(data->dbgdev_wq, (data->rd != data->wr)))
			return -ERESTARTSYS;
	}

	if (data->rd != data->wr) {
		count = buf_size > data->msg[data->rd].size ? data->msg[data->rd].size : buf_size;
		copy_to_user(buf, data->msg[data->rd].buf, count);
		data->rd = (data->rd + 1) & MSG_NUM_MASK;
		TRACE("dbgdev: return %d bytes on read.\n", count);
		return count;
		
	} else {
		return 0;
	}
}

static ssize_t device_write(struct file *f, const char* buf, size_t buf_size, loff_t* off)
{
	int start_pos = 0;
	int count;
	int warn_level = 0;
	dbg_data *data	= f->private_data;
	//if no emergency level is specified, use 0 as default.
	
	if (buf_size < 3) {
		data->msg[data->wr].buf[0] = '<';
		data->msg[data->wr].buf[1] = '0';
		data->msg[data->wr].buf[2] = '>';
		warn_level = 0;
		start_pos = 3;
		
	} else {
		copy_from_user(data->msg[data->wr].buf, buf, 3);
		if ((data->msg[data->wr].buf[0] != '<') || (data->msg[data->wr].buf[2] != '>')) {
			data->msg[data->wr].buf[0] = '<';
			data->msg[data->wr].buf[1] = '0';
			data->msg[data->wr].buf[2] = '>';
			warn_level = 0;
			start_pos = 3;
		} else {
			warn_level = data->msg[data->wr].buf[1] - '0';
			if (warn_level < 0 || warn_level > 7)
				warn_level = 0; //illegal warn-level value
			start_pos = 0;
		}
	}
 	
	// throw away.
	if (warn_level < data->warn_level) 
		return buf_size;
	
	count = buf_size > (MSG_BUF_SIZE - start_pos) ? (MSG_BUF_SIZE - start_pos) : buf_size;
	copy_from_user(data->msg[data->wr].buf+start_pos, buf, count);
	count += start_pos;

	// add \r\n
	// we have reserved 4 bytes in the end of buf, so we don't worry about buffer overflow
	if (data->msg[data->wr].buf[count-1] != '\r' &&  data->msg[data->wr].buf[count-1] != '\n') {
		count += 2;
	} else if (data->msg[data->wr].buf[count-2] != '\r' 
			&& data->msg[data->wr].buf[count-2] != '\n') {
		count += 1;
	}
	
	data->msg[data->wr].buf[count-1] = '\n';
	data->msg[data->wr].buf[count-2] = '\r';
	
	data->msg[data->wr].size = count;

	data->wr = ((data->wr + 1) & MSG_NUM_MASK);

	wake_up_interruptible(&data->dbgdev_wq);
	
	TRACE("%s write %d bytes\n", DEVICE_NAME, count);
	
	// if the buffer user sent is too long, just truncate it.
	if (data->output_to_console)
		return 0;	//tell the caller to output to serial port.
	else
		return buf_size;
}

static int device_open(struct inode *node, struct file *f)
{
	ref_cnt++;

	printk("%s opened(count = %d)\n", DEVICE_NAME, ref_cnt);
	f->private_data = &all_dbg_data;

	return 0;
}

static int device_release(struct inode *node, struct file *f)
{
	ref_cnt--;
	printk("%s released(count = %d)\n", DEVICE_NAME, ref_cnt);
	return 0;
}

static int device_ioctl(struct inode *node, struct file *f, unsigned int cmd, unsigned long value)
{
	dbg_data *data	= f->private_data;
	
	printk("dbgdev ioctl: cmd %ld, val %ld\n", cmd, value);

	switch (cmd) {
		case 0:
			data->warn_level = value ; //larger than 7 menas show  nothing
			printk("dbgdev: set warn level to %d\n", data->warn_level);
			break;
		case 1:
			data->output_to_console = value; //0: no, 1: yes
			printk("dbgdev: outputing to console %s\n", value?"enabled":"disabled");
		default:
			break;
	}
	
	return 0;
}

module_init(dbgdev_init);
module_exit(dbgdev_cleanup);


