/*
 * Handle mapping of the flash memory access routines
 * on MG6000 based devices.
 *
 * Copyright(C) 2004 Deng Xuejie, GaoKe Communications Technology, 
 * <dengxuejie@gktel.com.cn>
 *
 */

#include <linux/errno.h>
#include <linux/module.h>
#include <linux/types.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <asm/io.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/map.h>
#include <linux/mtd/partitions.h>


#define WINDOW_ADDR 0xff000000

#ifdef MG6002_SERIES

#if defined(RADON_MG6002_ET)

#define WINDOW_SIZE 0x00400000

#define NUM_PARTITIONS 5

/* partition_info gives details on the logical partitions that the split the
 * single flash device into. If the size if zero we use up to the end of the
 * device. */
static struct mtd_partition partition_info[]={
	/*{.name   = "bootloader",
	 .offset = 0,
	 .size   = 0x00010000, },*/

	{.name   = "environment",
	 .offset = 0x00010000,
	 .size   = 0x00010000, },

	{ .name  = "linux",
	 .offset = 0x00020000,
	 .size   = 0x000a0000, },

	{ .name  = "rootfs1",
	 .offset = 0x000c0000,
	 .size   = 0x00190000, },

	{ .name  = "rootfs2",
	 .offset = 0x00250000,
	 .size   = 0x00190000, },

	{.name	 = "paramter",
	 .offset = 0x003e0000,
	 .size	 = 0x00020000, } 
};

#elif defined(RADON_MG6002_VPN)

#define WINDOW_SIZE 0x00400000

#define NUM_PARTITIONS 5

/* partition_info gives details on the logical partitions that the split the
 * single flash device into. If the size if zero we use up to the end of the
 * device. */
static struct mtd_partition partition_info[]={
	/*{.name   = "bootloader",
	 .offset = 0,
	 .size   = 0x00010000, },*/

	{.name   = "environment",
	 .offset = 0x00010000,
	 .size   = 0x00010000, },

	{ .name  = "linux",
	 .offset = 0x00020000,
	 .size   = 0x000a0000, },

	{ .name  = "rerseved",
	 .offset = 0x000c0000,
	 .size   = 0x00120000, },

	{.name   = "paramter",
	 .offset = 0x001e0000,
	 .size   = 0x00020000, },

	{ .name  = "rootfs",
	 .offset = 0x00200000,
	 .size   = 0x00200000, }
};

#else // #if defined(RADON_MG6002_ET)

#define WINDOW_SIZE 0x00200000

#define NUM_PARTITIONS 4

/* partition_info gives details on the logical partitions that the split the
 * single flash device into. If the size if zero we use up to the end of the
 * device. */
static struct mtd_partition partition_info[]={
	/*{.name   = "bootloader",
	 .offset = 0,
	 .size   = 0x00010000, },*/

	{.name   = "environment",
	 .offset = 0x00010000,
	 .size   = 0x00010000, },

	{ .name  = "linux",
	 .offset = 0x00020000,
	 .size   = 0x00080000, },

	{ .name  = "rootfs",
	 .offset = 0x000a0000,
	 .size   = 0x00150000, },

	{.name   = "paramter",
	 .offset = 0x001f0000,
	 .size   = 0x00010000, }
};

#endif

#else // #ifdef MG6002_SERIES

#if defined(RADON_MG6008_ET)

#define WINDOW_SIZE 0x00400000
  
#define NUM_PARTITIONS 7
  
  /* partition_info gives details on the logical partitions that the split the
   * single flash device into. If the size if zero we use up to the end of the
   * device. */
  static struct mtd_partition partition_info[]={
	  /*{.name   = "bootloader",
	   .offset = 0,
	   .size   = 0x00020000, },*/
  
	  {.name   = "environment",
	   .offset = 0x00020000,
	   .size   = 0x00020000, },
  
	  { .name  = "linux",	  // kernel
	   .offset = 0x00040000, 
	   .size   = 0x000a0000, },
  
	  { .name  = "dsp1", 	  // dsp
	   .offset = 0x000e0000, 
	   .size   = 0x00020000, },

	  { .name  = "dsp2", 	  // dsp
	   .offset = 0x00100000, 
	   .size   = 0x00020000, },

	  { .name  = "rootfs1",	  // app
	   .offset = 0x00120000, 
	   .size   = 0x00160000, },

	  { .name  = "rootfs2",	  // app
	   .offset = 0x00280000, 
	   .size   = 0x00160000, },
  
	  {.name   = "paramter",
	   .offset = 0x003e0000,
	   .size   = 0x00020000, } 
  };

#else
    #if 0
	#define WINDOW_SIZE 0x00400000

	#define NUM_PARTITIONS 8

	/* partition_info gives details on the logical partitions that the split the
	 * single flash device into. If the size if zero we use up to the end of the
	 * device. */
	static struct mtd_partition partition_info[]={
		{.name   = "bootloader", // BOOT INFO
		 .offset = 0,
		 .size	 = 0x00020000, },
	
		{.name	 = "environment",
		 .offset = 0x00020000,
		 .size	 = 0x00020000, },

		{ .name  = "linux",     // kernel
		 .offset = 0x00040000, 
		 .size	 = 0x000a0000, },

		{ .name  = "dsp",       // dsp
		 .offset = 0x000e0000, 
		 .size	 = 0x00040000, },

		{ .name  = "voiceprompt",
		 .offset = 0x00120000,
		 .size	 = 0x00020000, },

		{ .name  = "reserve",
		 .offset = 0x00140000,
		 .size	 = 0x000a0000, },

		{.name	 = "paramter",
		 .offset = 0x001e0000,
		 .size	 = 0x00020000, },

		{ .name  = "rootfs",    // app
		 .offset = 0x00200000, 
		 .size	 = 0x00200000, }	
	};
    #endif
    
	#define WINDOW_SIZE 0x00400000

	#define NUM_PARTITIONS 7

	/* partition_info gives details on the logical partitions that the split the
	 * single flash device into. If the size if zero we use up to the end of the
	 * device. */
#if 0
	static struct mtd_partition partition_info[]={
		/*{.name   = "bootloader", // BOOT INFO
		 .offset = 0,
		 .size	 = 0x00010000, },*/
	
		{.name	 = "bootinfo",
		 .offset = 0x00010000,
		 .size	 = 0x00020000, },

		{ .name  = "linux",     // kernel
		 .offset = 0x00020000, 
		 .size	 = 0x00120000, },

		{ .name  = "romfs",       // romfs
		 .offset = 0x00120000, 
		 .size	 = 0x003f0000, },
		
		{ .name  = "rootfs",    // rever
		 .offset = 0x003f0000, 
		 .size	 = 0x00400000, }	
	};
#else
    	static struct mtd_partition partition_info[]={
		/*{.name   = "bootloader", // BOOT INFO
		 .offset = 0,
		 .size	 = 0x0000A000, },*/
	
		{.name	 = "parameters",
		 .offset = 0x0000A000,
		 .size	 = 0x00002000, },
		 
        {.name   = "reversed1",
         .offset = 0x0000C000,
         .size   = 0x00002000, },
         
 		{.name	 = "reversed2",
		 .offset = 0x0000E000,
		 .size	 = 0x00002000, },
		 
  		{.name	 = "bootinfo",
		 .offset = 0x00010000,
		 .size	 = 0x00010000, },

		{ .name  = "kernel",     // kernel
		 .offset = 0x00020000, 
		 .size	 = 0x00100000, },

		{ .name  = "romfs",       // romfs
		 .offset = 0x00120000, 
		 .size	 = 0x002D0000, },
		
		{ .name  = "reversed3",    // rever
		 .offset = 0x003F0000, 
		 .size	 = 0x00010000, }	
	};
#endif
#endif

#endif

static struct mtd_info *mymtd;

struct map_info mg6000_map = {
	.name = "MG6000 flash",
	.size = WINDOW_SIZE,
	.phys = WINDOW_ADDR,
	.buswidth = 1,
};

int __init init_mg6000(void)
{
	printk(KERN_NOTICE "Gaoke MG6000 flash device: 0x%x at 0x%x\n", WINDOW_SIZE, WINDOW_ADDR);
	mg6000_map.virt = (unsigned long)ioremap(WINDOW_ADDR, WINDOW_SIZE);

	if (!mg6000_map.virt) {
		printk("Failed to ioremap\n");
		return -EIO;
	}
	simple_map_init(&mg6000_map);

	mymtd = do_map_probe("cfi_probe", &mg6000_map);
	//mymtd = do_map_probe("jedec_probe", &mg6000_map);
	if (mymtd) {
		mymtd->owner = THIS_MODULE;
		add_mtd_device(mymtd);
        add_mtd_partitions(mymtd, partition_info, NUM_PARTITIONS);
		return 0;
	}

	iounmap((void *)mg6000_map.virt);
	return -ENXIO;
}

static void __exit cleanup_mg6002(void)
{
	if (mymtd) {
		del_mtd_device(mymtd);
		map_destroy(mymtd);
	}
	if (mg6000_map.virt) {
		iounmap((void *)mg6000_map.virt);
		mg6000_map.virt = 0;
	}
}

module_init(init_mg6000);
module_exit(cleanup_mg6002);

MODULE_AUTHOR("Deng Xuejie <Deng Xuejie@denxuejie@gktel.com.cn>");
MODULE_DESCRIPTION("MTD map driver for Gaoke MG6000 board");
MODULE_LICENSE("GPL");
