/*led.c غ 2009.11.27*/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/string.h>
#include <linux/timer.h>
#include <asm/io.h>
#include <asm/system.h>

#include <linux/types.h> 
#include <linux/fs.h> 
#include <linux/mm.h> 
#include <linux/errno.h> 
#include <asm/segment.h> 
#include <asm/uaccess.h>


#define Base_Addr   0xFFF00000
#define GPIO_CFG        (*((volatile unsigned long *)(Base_Addr+0x83000)))
#define GPIO_DIR        (*((volatile unsigned long *)(Base_Addr+0x83004)))
#define GPIO_DATAOUT    (*((volatile unsigned long *)(Base_Addr+0x83008)))


#define LED_MAJOR  113

/*IO CMD*/
#define LED_RUN   12 /*usb device id*/
#define LED_2G    15
#define LED_3G    16

#define LEN_RUN_ON  20
#define LEN_RUN_OFF 21
#define LEN_2G_ON   22
#define LEN_2G_OFF  23
#define LEN_3G_ON   24
#define LEN_3G_OFF  25

#define INIT_PARA   30

#define PPP_STAT_IS_UP 40
#define PPP_STAT_IS_DOWN 41
#define GET_PPP_STAT    42
#define ENABLE_LAN_IDLE_OFFLINE 43
#define DISABLE_LAN_IDLE_OFFLINE 44

typedef struct struct_led_status
{
    unsigned long value;
}led_status;

static led_status led_run_staus;
static led_status led_2G_staus;
static led_status led_3G_staus;

static int ppp_stat;

static int lan_idle_offline;


static void init_led(void)
{
    GPIO_CFG &= ~(3 << 12);
    GPIO_DIR |= (1 << LED_3G);  /*net red*/
    GPIO_DIR |= (1 << LED_2G);  /*net green*/
    
    GPIO_CFG &= ~(3 << 6);
    GPIO_DIR |= (1 << LED_RUN);  /*run*/
}

static void led_on(int type)
{    
    GPIO_DATAOUT &= ~(1 << type);
}

static void led_off(int type)
{
    GPIO_DATAOUT |= (1 << type);
}



static int open_ops(struct inode *inode, struct file *filp)
{ 
    return 0;

} 


void set_ppp_stat(int type)
{
    ppp_stat = type;
}

int get_ppp_stat(void)
{
    return ppp_stat;
}

int get_lan_idle_offline(void)
{
    return lan_idle_offline;

}
    
void set_lan_idle_offline(int type)
{
    lan_idle_offline = type;
}

static int read_ops(struct file *filp, char *buf, size_t size, loff_t *loffp)
{
    unsigned char msg[30];
    int i = 0;
    memcpy(msg+i, &led_run_staus, sizeof(led_status));
    i += sizeof(led_status);
    memcpy(msg+i, &led_2G_staus, sizeof(led_status));
    i += sizeof(led_status);
    memcpy(msg+i, &led_3G_staus, sizeof(led_status));
    i += sizeof(led_status);
    copy_to_user(buf, msg, i);
    return i;
} 
 
static int write_ops(struct file *filp, const char *buf, size_t size, loff_t *loffp)
{ 
    return 0; 
} 
static int ioctl_ops(struct inode *inode, struct file *filp, unsigned int cmd, unsigned long arg)
{
    unsigned long data = 0;
    static unsigned long ppp_status;
    copy_from_user(&data, (void *)arg, sizeof(unsigned long));
    //printk("ioctl...0x%08X\n", data);
    data = ((cmd<<24)|(data&0x00FFFFFF));
    switch (cmd)
    {
        case INIT_PARA:            
            led_run_staus.value=0x0C0107D0;
            led_2G_staus.value=0x0F0007D0;
            led_3G_staus.value=0x100007D0;
            init_led();
            break;
        case LED_RUN:
            led_run_staus.value = data;
            break;
        case LED_2G: 
            led_2G_staus.value = data;  
            break;    
        case LED_3G:
            led_3G_staus.value = data;
            break;
        case LEN_RUN_ON:
            led_on(LED_RUN);
            break;
        case LEN_RUN_OFF:
            led_off(LED_RUN);
            break;
        case LEN_2G_ON:
            led_on(LED_2G);
            break;
        case LEN_2G_OFF:
            led_off(LED_2G);
            break;
        case LEN_3G_ON:
            led_on(LED_3G);
            break;
        case LEN_3G_OFF:
            led_off(LED_3G);
            break;
        case PPP_STAT_IS_UP:
        case PPP_STAT_IS_DOWN:
            set_ppp_stat(cmd-PPP_STAT_IS_UP);            
            break;
        case GET_PPP_STAT:
            ppp_status = get_ppp_stat();
            copy_to_user((void *)arg, &ppp_status, sizeof(int));
            set_ppp_stat(0);
            break;
        case ENABLE_LAN_IDLE_OFFLINE:
            set_lan_idle_offline(1);
            break;
        case DISABLE_LAN_IDLE_OFFLINE:
            set_lan_idle_offline(0);
            break;    
        default:
            printk("Error cmd %d\n", cmd);
            return -1;
    }    
    return 0;
}


static int release_ops(struct inode *inode, struct file *filp)
{ 
    return 0;
} 
 
struct file_operations led_fops = { 
    open: open_ops,
    read: read_ops,
    write: write_ops,
    ioctl: ioctl_ops,
    release: release_ops,
};  

static int __init led_init_module(void) 
{ 
    int result; 
    result = register_chrdev(LED_MAJOR, "led", &led_fops); 
    if (result < 0) 
    { 
        printk(KERN_INFO "register led failed\n"); 
        return result; 
    } 
    return 0; 
} 

static void __exit led_cleanup_module(void) 
{ 
    unregister_chrdev(LED_MAJOR, "led"); 
} 

module_init(led_init_module)
module_exit(led_cleanup_module)



